# -*- coding: utf-8 -*-
"""
简繁转换工具 - 按文件尾标过滤并转换为繁体中文
功能：
1. 遍历指定目录下的所有.txt文件
2. 检测文件名是否包含特定尾标（c3/c4/g1/g2/g3/gpt/kimi/grok/deep）
3. 将简体中文转换为繁体中文（台湾标准）
4. 生成新文件，在原尾标后添加".ft"标记
"""

import os
import re
import sys
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
from pathlib import Path
import threading

# ====================================================================
# 依赖检查
# ====================================================================

def check_dependencies():
    """检查并安装依赖"""
    try:
        import opencc
        return True, "依赖库已就绪"
    except ImportError:
        try:
            import subprocess
            subprocess.check_call([
                sys.executable, "-m", "pip", "install", 
                "opencc-python-reimplemented", "-i", "https://pypi.tuna.tsinghua.edu.cn/simple"
            ])
            return True, "依赖库安装成功"
        except Exception as e:
            return False, f"依赖库安装失败: {str(e)}\n请手动执行: pip install opencc-python-reimplemented"

# ====================================================================
# 简繁转换器
# ====================================================================

class ChineseConverter:
    def __init__(self):
        try:
            from opencc import OpenCC
            # 使用台湾标准（s2twp = Simplified to Traditional Taiwan with Phrases）
            self.converter = OpenCC('s2twp')
        except Exception as e:
            self.converter = None
            print(f"警告：OpenCC初始化失败 - {e}")
    
    def convert(self, text):
        """简体转繁体（台湾标准）"""
        if self.converter is None:
            return text
        try:
            return self.converter.convert(text)
        except Exception as e:
            print(f"转换失败：{e}")
            return text

# ====================================================================
# 文件尾标检测
# ====================================================================

# 目标文件尾标列表（不区分大小写）
TARGET_SUFFIXES = [
    'c3', 'c4', 'g1', 'g2', 'g3', 
    'gpt', 'kimi', 'grok', 'deep'
]

def detect_target_suffix(filename):
    """
    检测文件名中是否包含目标尾标
    示例：RTZ003_0001.金刚萨埵意修甚深伏藏目录.掘藏者林巴.g2.0f.txt
    返回：(是否匹配, 匹配的尾标, 文件名各部分)
    """
    # 移除.txt后缀
    if not filename.lower().endswith('.txt'):
        return False, None, None
    
    name_without_ext = filename[:-4]  # 移除.txt
    
    # 按点号分割文件名
    parts = name_without_ext.split('.')
    
    # 检查是否包含目标尾标
    for i, part in enumerate(parts):
        if part.lower() in TARGET_SUFFIXES:
            return True, part, parts
    
    return False, None, None

def rebuild_filename(parts, matched_suffix):
    """
    重建文件名，在匹配的尾标后添加.ft
    示例：['RTZ003_0001', '金刚萨埵...', 'g2', '0f'] 
         → RTZ003_0001.金刚萨埵....g2.ft.0f.txt
    """
    new_parts = []
    for part in parts:
        new_parts.append(part)
        if part.lower() == matched_suffix.lower():
            new_parts.append('ft')
    
    return '.'.join(new_parts) + '.txt'

# ====================================================================
# 文件处理
# ====================================================================

def process_file(input_file, log_callback):
    """处理单个文件"""
    input_file = Path(input_file)
    
    # 检测文件名是否包含目标尾标
    has_suffix, matched_suffix, parts = detect_target_suffix(input_file.name)
    
    if not has_suffix:
        log_callback(f"  ⏭️  未检测到目标尾标，跳过\n")
        return None
    
    log_callback(f"  🔍 检测到尾标：'{matched_suffix}'\n")
    
    # 读取文件
    encodings = ['utf-8', 'utf-8-sig', 'gbk', 'gb2312', 'big5']
    content = None
    used_encoding = None
    
    for enc in encodings:
        try:
            with open(input_file, 'r', encoding=enc) as f:
                content = f.read()
            used_encoding = enc
            break
        except (UnicodeDecodeError, UnicodeError):
            continue
    
    if content is None:
        log_callback(f"  ❌ 无法读取文件（编码问题）\n")
        return None
    
    # 初始化转换器
    converter = ChineseConverter()
    
    # 转换内容
    log_callback(f"  🔄 正在转换内容...\n")
    converted_content = converter.convert(content)
    
    # 转换文件名中的中文部分
    converted_parts = [converter.convert(part) for part in parts]
    
    # 重建文件名（在匹配的尾标后添加.ft）
    new_filename = rebuild_filename(converted_parts, matched_suffix)
    output_file = input_file.parent / new_filename
    
    # 写入文件
    try:
        with open(output_file, 'w', encoding='utf-8') as f:
            f.write(converted_content)
        log_callback(f"  ✅ 生成繁体版 → {new_filename}\n")
        return str(output_file)
    except Exception as e:
        log_callback(f"  ❌ 写入失败：{str(e)}\n")
        return None

# ====================================================================
# 图形界面
# ====================================================================

class SimplifiedToTraditionalGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("简繁转换工具 v2.0 - 按尾标过滤")
        self.root.geometry("800x650")
        
        # 标题
        title_label = tk.Label(
            root, 
            text="简体转繁体工具（台湾标准）",
            font=("Arial", 16, "bold"),
            fg="#2c3e50"
        )
        title_label.pack(pady=10)
        
        # 说明文字
        info_label = tk.Label(
            root,
            text="自动检测文件尾标并转换为台湾繁体中文 | 在原尾标后添加 .ft 标记",
            font=("Arial", 9),
            fg="#7f8c8d"
        )
        info_label.pack(pady=5)
        
        # 目标尾标显示区域
        suffix_frame = tk.LabelFrame(root, text="目标文件尾标列表", font=("Arial", 10, "bold"))
        suffix_frame.pack(pady=10, padx=20, fill=tk.X)
        
        suffix_text = tk.Label(
            suffix_frame,
            text=", ".join(TARGET_SUFFIXES),
            font=("Consolas", 9),
            fg="#e74c3c",
            wraplength=700,
            justify="left"
        )
        suffix_text.pack(padx=10, pady=10)
        
        # 示例说明
        example_frame = tk.Frame(root, bg="#e8f5e9", bd=1, relief=tk.SOLID)
        example_frame.pack(pady=10, padx=20, fill=tk.X)
        
        tk.Label(
            example_frame,
            text="📝 示例：\n"
                 "原文件：RTZ003_0001.金刚萨埵意修甚深伏藏目录.掘藏者林巴.g2.0f.txt\n"
                 "新文件：RTZ003_0001.金剛薩埵意修甚深伏藏目錄.掘藏者林巴.g2.ft.0f.txt",
            font=("Consolas", 8),
            bg="#e8f5e9",
            fg="#2e7d32",
            justify="left"
        ).pack(pady=8, padx=10, anchor="w")
        
        # 提示信息
        notice_frame = tk.Frame(root, bg="#fff3cd", bd=1, relief=tk.SOLID)
        notice_frame.pack(pady=10, padx=20, fill=tk.X)
        
        tk.Label(
            notice_frame,
            text="⚠️ 只有文件名包含上述尾标的文件才会被转换",
            font=("Arial", 9),
            bg="#fff3cd",
            fg="#856404"
        ).pack(pady=8)
        
        # 选择目录按钮
        self.select_button = tk.Button(
            root,
            text="📂 选择目录开始转换",
            command=self.select_directory,
            font=("Arial", 12),
            bg="#3498db",
            fg="white",
            padx=20,
            pady=10,
            cursor="hand2"
        )
        self.select_button.pack(pady=15)
        
        # 进度显示
        self.progress_label = tk.Label(
            root,
            text="等待选择目录...",
            font=("Arial", 10),
            fg="#95a5a6"
        )
        self.progress_label.pack(pady=5)
        
        # 日志区域
        log_frame = tk.Frame(root)
        log_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        tk.Label(log_frame, text="处理日志:", font=("Arial", 10)).pack(anchor="w")
        
        self.log_text = scrolledtext.ScrolledText(
            log_frame,
            wrap=tk.WORD,
            font=("Consolas", 9),
            bg="#ecf0f1",
            fg="#2c3e50"
        )
        self.log_text.pack(fill=tk.BOTH, expand=True)
        
        # 底部按钮
        button_frame = tk.Frame(root)
        button_frame.pack(pady=10)
        
        tk.Button(
            button_frame,
            text="测试转换",
            command=self.test_conversion,
            font=("Arial", 10),
            bg="#2ecc71",
            fg="white",
            padx=15,
            pady=5
        ).pack(side=tk.LEFT, padx=5)
        
        tk.Button(
            button_frame,
            text="清空日志",
            command=self.clear_log,
            font=("Arial", 10),
            bg="#e74c3c",
            fg="white",
            padx=15,
            pady=5
        ).pack(side=tk.LEFT, padx=5)
    
    def log(self, message):
        """添加日志"""
        self.log_text.insert(tk.END, message)
        self.log_text.see(tk.END)
        self.root.update()
    
    def clear_log(self):
        """清空日志"""
        self.log_text.delete(1.0, tk.END)
    
    def test_conversion(self):
        """测试转换"""
        self.clear_log()
        self.log("🧪 测试模式\n\n")
        
        test_files = [
            "RTZ003_0001.金刚萨埵意修甚深伏藏目录.掘藏者林巴.g2.0f.txt",
            "T01n0009_Manussisi Sutta.pali.gpt.txt",
            "佛说阿弥陀经.c3.notes.txt",
            "普通文件.没有尾标.txt",
        ]
        
        converter = ChineseConverter()
        
        for filename in test_files:
            self.log(f"【测试文件】{filename}\n")
            
            has_suffix, matched_suffix, parts = detect_target_suffix(filename)
            
            if has_suffix:
                self.log(f"✅ 检测到尾标：'{matched_suffix}'\n")
                
                # 转换文件名
                converted_parts = [converter.convert(part) for part in parts]
                new_filename = rebuild_filename(converted_parts, matched_suffix)
                self.log(f"新文件名：{new_filename}\n")
            else:
                self.log("❌ 未检测到目标尾标，跳过\n")
            
            self.log("\n")
    
    def select_directory(self):
        """选择目录并开始处理"""
        directory = filedialog.askdirectory(title="选择包含文本文件的目录")
        
        if not directory:
            return
        
        # 在新线程中处理
        thread = threading.Thread(
            target=self.process_directory, 
            args=(directory,)
        )
        thread.daemon = True
        thread.start()
    
    def process_directory(self, directory):
        """处理目录"""
        self.select_button.config(state=tk.DISABLED)
        self.clear_log()
        
        self.log(f"📂 选择的目录: {directory}\n")
        self.log(f"🔍 目标尾标: {', '.join(TARGET_SUFFIXES)}\n")
        self.log("=" * 70 + "\n\n")
        
        # 查找所有.txt文件
        txt_files = list(Path(directory).rglob('*.txt'))
        
        # 排除已经是.ft.txt的文件
        txt_files = [f for f in txt_files if '.ft.' not in f.name.lower()]
        
        if not txt_files:
            self.log("⚠️ 未找到任何.txt文件\n")
            messagebox.showwarning("未找到文件", f"目录 {directory} 下没有找到.txt文件")
            self.select_button.config(state=tk.NORMAL)
            return
        
        self.log(f"找到 {len(txt_files)} 个文本文件\n\n")
        
        total_processed = 0
        total_skipped = 0
        
        for i, txt_file in enumerate(txt_files, 1):
            self.progress_label.config(text=f"处理进度: {i}/{len(txt_files)}")
            
            rel_path = txt_file.relative_to(directory)
            self.log(f"[{i}/{len(txt_files)}] {rel_path}\n")
            
            result = process_file(txt_file, self.log)
            
            if result:
                total_processed += 1
            else:
                total_skipped += 1
            
            self.log("\n")
        
        self.log("=" * 70 + "\n")
        self.log(f"🎉 全部完成！\n")
        self.log(f"   检查文件: {len(txt_files)} 个\n")
        self.log(f"   已转换: {total_processed} 个\n")
        self.log(f"   已跳过: {total_skipped} 个\n")
        
        self.progress_label.config(text="转换完成！")
        self.select_button.config(state=tk.NORMAL)
        
        messagebox.showinfo(
            "转换完成", 
            f"检查了 {len(txt_files)} 个文件\n转换了 {total_processed} 个文件\n跳过了 {total_skipped} 个文件"
        )

# ====================================================================
# 主程序
# ====================================================================

def main():
    # 检查依赖
    success, message = check_dependencies()
    
    if not success:
        root = tk.Tk()
        root.withdraw()
        messagebox.showerror("依赖错误", message)
        sys.exit(1)
    
    # 创建主窗口
    root = tk.Tk()
    app = SimplifiedToTraditionalGUI(root)
    root.mainloop()

if __name__ == "__main__":
    main()
